import XCTest
@testable import death_app_Watch_App

final class SurveyValidationTests: XCTestCase {
    
    var surveyValidator: SurveyValidator!
    
    override func setUpWithError() throws {
        try super.setUpWithError()
        surveyValidator = SurveyValidator()
    }
    
    override func tearDownWithError() throws {
        surveyValidator = nil
        try super.tearDownWithError()
    }
    
    // MARK: - Age Validation Tests
    
    func testValidAgeRange() {
        XCTAssertTrue(surveyValidator.isValidAge(25))
        XCTAssertTrue(surveyValidator.isValidAge(65))
        XCTAssertTrue(surveyValidator.isValidAge(100))
    }
    
    func testInvalidAgeRange() {
        XCTAssertFalse(surveyValidator.isValidAge(-1))
        XCTAssertFalse(surveyValidator.isValidAge(0))
        XCTAssertFalse(surveyValidator.isValidAge(17))
        XCTAssertFalse(surveyValidator.isValidAge(121))
        XCTAssertFalse(surveyValidator.isValidAge(200))
    }
    
    // MARK: - Height Validation Tests
    
    func testValidHeightMetric() {
        XCTAssertTrue(surveyValidator.isValidHeight(150, unit: .centimeters)) // 150cm
        XCTAssertTrue(surveyValidator.isValidHeight(180, unit: .centimeters)) // 180cm
        XCTAssertTrue(surveyValidator.isValidHeight(220, unit: .centimeters)) // 220cm
    }
    
    func testValidHeightImperial() {
        XCTAssertTrue(surveyValidator.isValidHeight(60, unit: .inches)) // 5'0"
        XCTAssertTrue(surveyValidator.isValidHeight(72, unit: .inches)) // 6'0"
        XCTAssertTrue(surveyValidator.isValidHeight(84, unit: .inches)) // 7'0"
    }
    
    func testInvalidHeight() {
        XCTAssertFalse(surveyValidator.isValidHeight(50, unit: .centimeters)) // Too short
        XCTAssertFalse(surveyValidator.isValidHeight(300, unit: .centimeters)) // Too tall
        XCTAssertFalse(surveyValidator.isValidHeight(30, unit: .inches)) // Too short
        XCTAssertFalse(surveyValidator.isValidHeight(120, unit: .inches)) // Too tall
    }
    
    // MARK: - Weight Validation Tests
    
    func testValidWeightKilograms() {
        XCTAssertTrue(surveyValidator.isValidWeight(50, unit: .kilograms))
        XCTAssertTrue(surveyValidator.isValidWeight(80, unit: .kilograms))
        XCTAssertTrue(surveyValidator.isValidWeight(150, unit: .kilograms))
    }
    
    func testValidWeightPounds() {
        XCTAssertTrue(surveyValidator.isValidWeight(100, unit: .pounds))
        XCTAssertTrue(surveyValidator.isValidWeight(180, unit: .pounds))
        XCTAssertTrue(surveyValidator.isValidWeight(300, unit: .pounds))
    }
    
    func testInvalidWeight() {
        XCTAssertFalse(surveyValidator.isValidWeight(20, unit: .kilograms)) // Too light
        XCTAssertFalse(surveyValidator.isValidWeight(500, unit: .kilograms)) // Too heavy
        XCTAssertFalse(surveyValidator.isValidWeight(50, unit: .pounds)) // Too light
        XCTAssertFalse(surveyValidator.isValidWeight(800, unit: .pounds)) // Too heavy
    }
    
    // MARK: - BMI Validation Tests
    
    func testBMICalculation() {
        let bmi1 = surveyValidator.calculateBMI(weight: 70, height: 175, weightUnit: .kilograms, heightUnit: .centimeters)
        XCTAssertEqual(bmi1, 22.86, accuracy: 0.01)
        
        let bmi2 = surveyValidator.calculateBMI(weight: 150, height: 68, weightUnit: .pounds, heightUnit: .inches)
        XCTAssertEqual(bmi2, 22.8, accuracy: 0.1)
    }
    
    func testBMICategories() {
        XCTAssertEqual(surveyValidator.getBMICategory(bmi: 17.5), .underweight)
        XCTAssertEqual(surveyValidator.getBMICategory(bmi: 22.0), .normal)
        XCTAssertEqual(surveyValidator.getBMICategory(bmi: 27.0), .overweight)
        XCTAssertEqual(surveyValidator.getBMICategory(bmi: 32.0), .obese)
    }
    
    // MARK: - Smoking Validation Tests
    
    func testSmokingPackYearsCalculation() {
        let packYears1 = surveyValidator.calculatePackYears(cigarettesPerDay: 20, yearsSmoked: 10)
        XCTAssertEqual(packYears1, 10.0, accuracy: 0.01) // 1 pack/day for 10 years = 10 pack-years
        
        let packYears2 = surveyValidator.calculatePackYears(cigarettesPerDay: 10, yearsSmoked: 20)
        XCTAssertEqual(packYears2, 10.0, accuracy: 0.01) // 0.5 pack/day for 20 years = 10 pack-years
        
        let packYears3 = surveyValidator.calculatePackYears(cigarettesPerDay: 0, yearsSmoked: 0)
        XCTAssertEqual(packYears3, 0.0)
    }
    
    func testSmokingRiskCategory() {
        XCTAssertEqual(surveyValidator.getSmokingRiskCategory(packYears: 0), .never)
        XCTAssertEqual(surveyValidator.getSmokingRiskCategory(packYears: 5), .light)
        XCTAssertEqual(surveyValidator.getSmokingRiskCategory(packYears: 15), .moderate)
        XCTAssertEqual(surveyValidator.getSmokingRiskCategory(packYears: 25), .heavy)
        XCTAssertEqual(surveyValidator.getSmokingRiskCategory(packYears: 50), .extreme)
    }
    
    // MARK: - Exercise Validation Tests
    
    func testExerciseValidation() {
        XCTAssertTrue(surveyValidator.isValidExerciseMinutesPerWeek(0))
        XCTAssertTrue(surveyValidator.isValidExerciseMinutesPerWeek(150))
        XCTAssertTrue(surveyValidator.isValidExerciseMinutesPerWeek(300))
        XCTAssertTrue(surveyValidator.isValidExerciseMinutesPerWeek(600))
        
        XCTAssertFalse(surveyValidator.isValidExerciseMinutesPerWeek(-10))
        XCTAssertFalse(surveyValidator.isValidExerciseMinutesPerWeek(2000)) // Unrealistic
    }
    
    func testExerciseRiskCategory() {
        XCTAssertEqual(surveyValidator.getExerciseRiskCategory(minutesPerWeek: 0), .sedentary)
        XCTAssertEqual(surveyValidator.getExerciseRiskCategory(minutesPerWeek: 75), .insufficient)
        XCTAssertEqual(surveyValidator.getExerciseRiskCategory(minutesPerWeek: 175), .adequate)
        XCTAssertEqual(surveyValidator.getExerciseRiskCategory(minutesPerWeek: 350), .optimal)
    }
    
    // MARK: - Alcohol Validation Tests
    
    func testAlcoholConsumptionValidation() {
        XCTAssertTrue(surveyValidator.isValidAlcoholDrinksPerWeek(0))
        XCTAssertTrue(surveyValidator.isValidAlcoholDrinksPerWeek(7))
        XCTAssertTrue(surveyValidator.isValidAlcoholDrinksPerWeek(14))
        XCTAssertTrue(surveyValidator.isValidAlcoholDrinksPerWeek(21))
        
        XCTAssertFalse(surveyValidator.isValidAlcoholDrinksPerWeek(-1))
        XCTAssertFalse(surveyValidator.isValidAlcoholDrinksPerWeek(100)) // Unrealistic
    }
    
    func testAlcoholRiskCategory() {
        XCTAssertEqual(surveyValidator.getAlcoholRiskCategory(drinksPerWeek: 0), .none)
        XCTAssertEqual(surveyValidator.getAlcoholRiskCategory(drinksPerWeek: 5), .low)
        XCTAssertEqual(surveyValidator.getAlcoholRiskCategory(drinksPerWeek: 10), .moderate)
        XCTAssertEqual(surveyValidator.getAlcoholRiskCategory(drinksPerWeek: 20), .high)
        XCTAssertEqual(surveyValidator.getAlcoholRiskCategory(drinksPerWeek: 30), .extreme)
    }
    
    // MARK: - Medical History Validation Tests
    
    func testMedicalConditionsValidation() {
        let validConditions = ["diabetes", "hypertension", "heart_disease", "cancer"]
        let invalidConditions = ["", "unknown_condition", "123", "!@#"]
        
        XCTAssertTrue(surveyValidator.areValidMedicalConditions(validConditions))
        XCTAssertFalse(surveyValidator.areValidMedicalConditions(invalidConditions))
        XCTAssertFalse(surveyValidator.areValidMedicalConditions(validConditions + invalidConditions))
    }
    
    // MARK: - Complete Survey Validation Tests
    
    func testCompleteSurveyValidation() {
        let validSurvey = SurveyData(
            age: 35,
            gender: .male,
            height: 180,
            heightUnit: .centimeters,
            weight: 75,
            weightUnit: .kilograms,
            smokingStatus: .never,
            exerciseMinutesPerWeek: 200,
            alcoholDrinksPerWeek: 5,
            medicalConditions: ["hypertension"],
            familyHistory: ["heart_disease"]
        )
        
        let result = surveyValidator.validateSurvey(validSurvey)
        XCTAssertTrue(result.isValid)
        XCTAssertTrue(result.errors.isEmpty)
    }
    
    func testIncompleteSurveyValidation() {
        let invalidSurvey = SurveyData(
            age: 17, // Invalid - too young
            gender: .male,
            height: 50, // Invalid - too short
            heightUnit: .centimeters,
            weight: 500, // Invalid - too heavy
            weightUnit: .kilograms,
            smokingStatus: .never,
            exerciseMinutesPerWeek: -10, // Invalid - negative
            alcoholDrinksPerWeek: 100, // Invalid - too much
            medicalConditions: ["invalid_condition"], // Invalid condition
            familyHistory: []
        )
        
        let result = surveyValidator.validateSurvey(invalidSurvey)
        XCTAssertFalse(result.isValid)
        XCTAssertGreaterThan(result.errors.count, 0)
        
        XCTAssertTrue(result.errors.contains { $0.contains("age") })
        XCTAssertTrue(result.errors.contains { $0.contains("height") })
        XCTAssertTrue(result.errors.contains { $0.contains("weight") })
        XCTAssertTrue(result.errors.contains { $0.contains("exercise") })
        XCTAssertTrue(result.errors.contains { $0.contains("alcohol") })
        XCTAssertTrue(result.errors.contains { $0.contains("medical") })
    }
    
    // MARK: - Edge Cases
    
    func testBoundaryValues() {
        // Test exact boundary values
        XCTAssertTrue(surveyValidator.isValidAge(18)) // Minimum valid age
        XCTAssertTrue(surveyValidator.isValidAge(120)) // Maximum valid age
        XCTAssertFalse(surveyValidator.isValidAge(17)) // Just below minimum
        XCTAssertFalse(surveyValidator.isValidAge(121)) // Just above maximum
    }
    
    func testNilAndEmptyValues() {
        XCTAssertTrue(surveyValidator.areValidMedicalConditions([])) // Empty array should be valid
        XCTAssertFalse(surveyValidator.areValidMedicalConditions([""])) // Array with empty string should be invalid
    }
    
    func testSpecialCharactersInText() {
        let conditionsWithSpecialChars = ["condition@123", "test!condition", "condition with spaces"]
        XCTAssertFalse(surveyValidator.areValidMedicalConditions(conditionsWithSpecialChars))
    }
}